/*
Battery Capacity Meter Firmware
Author: Mauro Grassi, February 2009 for Silicon Chip Publications.
Notes: by Mauro Grassi.

Important Note: Compilation Instructions for the C18 compiler:
- All optimizations ON (Procedural Abstraction 3 passes) with banking optimizer OFF!
- Multi Bank Large Data Model Small Code Model
(M.G.)

USB may lead to crashes of host computer if the bypass capacitor on the Vusb pin is not big enough in value. 
I have been using a 1uF and it works well. With a 100nF instead you will get constant
crashing of host PC when USB plugged in. 
Note that the datasheet recommends 220nF certainly don't use any smaller than this.

Versions up to 4.00 were on a breadboard with slightly different pin outs, especially for LCD and AN channels.
Versions from 4.00 are for the PC board.

Version 4.30: first version with full coherent RLE and USB functioning correctly (fixed a small bug in the goOutOfStandBy routine
was calling USBConnect() regardless of whether the USB was already connected.

*/

#include <p18cxxx.h>
#include "typedefs.h"                   
#include "io_cfg.h"
#include "usb.h"
#include "batterymeter.h"
#include <stdlib.h>
#include <math.h>

#if   defined(__18F4550)||defined(__18F4455)|| \
	  defined(__18F2550)||defined(__18F2455)|| \
      defined(__18F4553)||defined(__18F4458)|| \
      defined(__18F2553)||defined(__18F2458)

#pragma config PLLDIV   = 5       		// (20 MHz input)
#pragma config CPUDIV   = OSC1_PLL2		//
#pragma config USBDIV   = 2       		// Clock source from 96MHz PLL/2
#pragma config FOSC     = HSPLL_HS		//
#pragma config FCMEN    = OFF
#pragma config IESO     = OFF
#pragma config PWRT     = ON
#pragma config BOR      = ON
#pragma config BORV     = 3
#pragma config VREGEN   = ON
#pragma config WDT      = OFF
#pragma config WDTPS    = 32768
#pragma config MCLRE    = OFF			// not using MCLR as reset pin = OFF
#pragma config LPT1OSC  = OFF			// you could enable this if you want.
#pragma config PBADEN   = OFF
#pragma config CCP2MX   = ON
#pragma config STVREN   = ON			// Stack Overflow Reset Enabled!!
#pragma config LVP      = OFF
//#pragma config ICPRT    = OFF       	// Dedicated In-Circuit Debug/Programming
#pragma config XINST    = OFF       	// Extended Instruction Set
#pragma config CP0      = OFF
#pragma config CP1      = OFF
#pragma config CP2      = OFF
#pragma config CP3      = OFF
#pragma config CPB      = OFF
#pragma config CPD      = OFF
#pragma config WRT0     = OFF
#pragma config WRT1     = OFF
#pragma config WRT2     = OFF
#pragma config WRT3     = OFF
#pragma config WRTB     = OFF      		// Boot Block Write Protection
#pragma config WRTC     = OFF
#pragma config WRTD     = OFF
#pragma config EBTR0    = OFF
#pragma config EBTR1    = OFF
#pragma config EBTR2    = OFF
#pragma config EBTR3    = OFF
#pragma config EBTRB    = OFF
//
#endif

// Declarations
extern int ReadEEPROM(int);
extern int WriteEEPROM(int, int);
byte translateKeyCode(byte);
byte translateNumberCode(byte);
void clearWithoutUpdate(void);
void updateLogChannelStrings(byte);
void displayDisplayMode(byte);
void USBConnect(void);
void saveSettings(void);
void restoreSettings(void);
void shiftGeneral(int, int, int, int, byte*);
void enterNumberString(byte, ram char*, int);
void doBeep(int);
void idF(int);
void restoreAllDefaults(void);
void declareFullCapacity(void);
void resetTimeOut(void);
void setFullCapacity(int);
void setVoltageRail50(int);
void setVoltageRail33(int);
void setThreshold(int);
void setPeukerts(int);
void setLogError(int);
void setLogPeriod(int);
void setChemistry(int);
void setMinVoltage(int);
void setMaxVoltage(int);
void setDividerLow(int);
void setCutOffVoltage(int);
void setAlarm(int);
void setDividerHigh(int);
void setChargingEfficiency(int);
void seeFirmwareVersion(int);
void seeUSBStatus(int);
void seeBatteryPercent(int);
void seeBatteryCapacityAH(int);
void seeBatteryVoltage(int);
void seeLoadCurrent(int);
void seeChargeCurrent(int);
void seeCircuitCurrent(int);
void seeNetBatteryDrain(int);
void seeBatteryTimeMinutes(int);
void setBeeperStatus(int);
void setTimeOutSeconds(int);
void setBrightnessPercent(int);
void setDifferentialGain(int);
void seeVoltageRail50(int);
void seeVUSBRail(int);
void seeCircuitVoltage(int);
void seeLoadDiffAmpVoltage(int);
void seeChargeDiffAmpVoltage(int);
void setShuntResistance(int);
void setCh1(int);
void setCh2(int);
void setCh3(int);
void setCh4(int);
void setSenseResistance(int);
void setBrightnessLevel(double);
void writeCellTypeString(byte);
rom MENU batterySettingsMenu;
rom MENU systemSettingsMenu;
rom MENU displayMenu;
rom MENU calibrationSettingsMenu;
rom MENU mainMenu;
rom MENU systemStatusMenu;
rom MENU* executeMenu(byte, rom MENU*);
void displayDisplayMode(byte);
// Global Variables
#pragma udata gpr0
// System Variables
byte scanLn;
byte scanCode;
byte scanRaw;
byte scanCounter;
byte displayMode;
byte keyBuffer[KEY_BUFFER_SIZE];
byte keyGetPtr;
byte keyPtr;
byte keyFull;
byte noKeys;
byte lcdScreen[LCD_SIZE];
byte oldlcdScreen[LCD_SIZE];
int lcdTransitionType;
byte reqReset;
char buffer[NUM_DIGITS+4];
byte tempbuffer[TEMPBUFF_SIZE];
byte roundingMode;
byte brightness;
byte setBrightness;
SAMPLE sample[NUM_SAMPLES-2];
double capacityAs;
double deltaCapacityAs;
double time;
byte totalUpdated;
unsigned int samplingCount;
byte samplingIndex;
byte alarmOn;
double logPeriod;
double logError;
double logPeriodCounter;
double alarmCapacity;
double cutOffVoltage;				// voltage below which the circuit cuts off power completely
double vusbRail;
double chargeDiffAmpVoltage;
double loadDiffAmpVoltage;
double circuitVoltage;				// in Volts
double batteryPercent;				// battery percent charge
double batteryCapacityAH;			// battery capacity in AH (Amp Hours)
double batteryVoltage;				// in Volts
double loadCurrent;					// in Amps
double circuitCurrent;				// in mAmps
double netCurrent;					// net current drain in Amps
double batteryMinutes;				// remaining Capacity
double chargeCurrent;				// charging current in A
double differentialGain;			// gain of the front end differential current sense chip
double shuntResistance;				// in milli Ohms
double senseResistance;				// in the circuit to measure the circuit current drain (in Ohms)
double timeOutSecondsCounter;
double timeOutInSeconds;
double brightnessPercent;
double chargingEfficiency;			// a percentage
double fullCapacity;				// in AH (Amp Hours)
double voltageRail33;				// +3.3V Rail (Volts)
double voltageRail50;				// +5V 	 Rail (Volts)
double dividerLow;					// voltage divider for the lower of the two battery voltages
double dividerHigh;					// voltage divider for the higher of the two battery voltages
double peukertsConstant;			// a constant
double minVoltage;					// voltage at which the battery can be considered fully discharged
double maxVoltage;					// voltage at which the battery can be considered fully charged (when trickle charging)
char   cellTypeString[10];			// cell chemistry string either "Nickel" or "Lead Acid"
char   yesNoUSBString[4];			// this string is either 'Yes' or 'No'
char   yesNoBeeperString[4];		// this string is either 'Yes' or 'No'
char   timeString[16];				// this is the time string in dXXhXXm format (days, hours, minutes)
double firmwareVersion;				// firmware version as double X.XX (2 decimal places are significant)
byte   beepOn;						// 1= beeper enabled 0= beeper disabled
byte cellType;
rom MENU* menuStack[MENU_STACK_SIZE];
byte menuStartIndexStack[MENU_STACK_SIZE];
byte menuPtr;
byte menuFull;
byte standBy;
byte enableStandByDetect;
// Data Logging Subsystem Data & Buffer
double 			RLEData[RLE_CHANNELS];
unsigned int 	RLEMultiplicity[RLE_CHANNELS];
byte RLEBuffer[HALF_RLE_BUFFER_SIZE];
byte RLEGetPtr;
byte RLEPtr;
byte RLEFull;
byte logChannel[RLE_CHANNELS];
char logChannelString[RLE_CHANNELS][8];
byte dummy;
#pragma udata usb5=0x4A0
byte RLEBuffer2[HALF_RLE_BUFFER_SIZE];
unsigned int timerDelay;
double currentThreshold;			// in Amps to go into standby mode
#pragma code

rom MENU calibrationSettingsMenu=
{
	7,
	{
		{ 0, { "Set +3.3V Rail Voltage (V)", setVoltageRail33, 	&voltageRail33, 	2, 		0 } },
		{ 0, { "Set +5.0V Rail Voltage (V)", setVoltageRail50,		&voltageRail50, 	2, 		0 } },
		{ 0, { "Calibrate Low Voltage Div.", setDividerLow, 		&dividerLow,		2, 		0 } },
		{ 0, { "Calibrate High Voltage Div.", setDividerHigh,		&dividerHigh, 		2, 		0 } },
		{ 0, { "Differential Amp. Gain", setDifferentialGain,  &differentialGain, 	2,		0 } },
		{ 0, { "Shunt Resistance (m\xF4)", setShuntResistance,   &shuntResistance, 	0,		0 } },
		{ 0, { "Circuit Sense Resistance(\xF4)", setSenseResistance,   &senseResistance, 	0,		0 } }
	}
};

rom MENU batterySettingsMenu=
{
	7,
	{
		{ 0, { "Declare Capacity Now Full", 	declareFullCapacity ,	0, 						0, 		0 } },
//		{ (MENU*)&systemStatusMenu,  		{ "Battery Status    ", 0, 0, 0, 0 } },
		{ 0, { "Full Battery Capacity (AH)",	setFullCapacity , 		&fullCapacity, 			1, 		0 } },
		{ 0, { "Peukert's Constant", 			setPeukerts,			&peukertsConstant, 		3, 		0 } },
		{ 0, { "Cell Chemistry", 				setChemistry, 			0, 						0, 		&cellTypeString[0] } },
		{ 0, { "Charging Efficiency (%)", 		setChargingEfficiency,	&chargingEfficiency,	1, 		0 } },
		{ 0, { "Min. Chrg. Voltage (V)", 	setMinVoltage, 			&minVoltage, 			1, 		0 } },
		{ 0, { "Max. Chrg. Voltage (V)", 	setMaxVoltage, 			&maxVoltage, 			1, 		0 } }
	}
};

rom MENU systemSettingsMenu=
{
	6,
	{ 
		{ (rom MENU*)0, { (rom char*)"Beeper Status", setBeeperStatus, 0, 0, &yesNoBeeperString[0] } },
		{ (rom MENU*)0, { (rom char*)"Shutdown Voltage (V)", setCutOffVoltage, &cutOffVoltage, 1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Low Cpty.Alarm (%)", setAlarm, &alarmCapacity, 0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Standby Current (A)", setThreshold, &currentThreshold, 3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Firmware Ver.", seeFirmwareVersion, &firmwareVersion, -2, 0 } },	// unsigned 2 decimal places
		{ (rom MENU*)0, { (rom char*)"USB Connected", seeUSBStatus, 0, 0, &yesNoUSBString[0] } },
	}
};

/*
rom MENU systemStatusMenu=
{
	8,
	{ 
		{ (rom MENU*)0, { (rom char*)"Batt. Charge (%)", seeBatteryPercent,    	&batteryPercent,		0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Batt. Capacity (AH)", seeBatteryCapacityAH, 	&batteryCapacityAH, 	0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Batt. Voltage (V)", seeBatteryVoltage, 	  	&batteryVoltage, 		1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Load Current (A)", seeLoadCurrent, 	  	&loadCurrent,			2, 0 } },
		{ (rom MENU*)0, { (rom char*)"Charge Current (A)", seeChargeCurrent, 	  	&chargeCurrent,			2, 0 } },
		{ (rom MENU*)0, { (rom char*)"Circuit Drain (mA)", seeCircuitCurrent, 	  	&circuitCurrent,		0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Net Battery Drain (A)", seeNetBatteryDrain,   	&netCurrent,			2, 0 } },
		{ (rom MENU*)0, { (rom char*)"Time Remaining (mins)", seeBatteryTimeMinutes,	&batteryMinutes,		0, 0 } },
	}
};
*/

rom MENU adcSystemStatusMenu=
{
	6,
	{ 
		{ (rom MENU*)0, { (rom char*)"+5.0V Rail Voltage (V)", 		seeVoltageRail50,    	&voltageRail50,			2, 0 } },
		{ (rom MENU*)0, { (rom char*)"+3.3V Rail Voltage (V)",	 	seeVUSBRail,       		&vusbRail,     			2, 0 } },
		{ (rom MENU*)0, { (rom char*)"Battery Voltage (V)", 		seeBatteryVoltage, 	  	&batteryVoltage, 		1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Circuit Voltage (V)", 		seeCircuitVoltage,		&circuitVoltage,		1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Load Diff. Amp Voltage (V)", 	seeLoadDiffAmpVoltage,	&loadDiffAmpVoltage,	3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Chrg.Diff. Amp Voltage (V)", 	seeChargeDiffAmpVoltage,&chargeDiffAmpVoltage,	3, 0 } }
	}
};

rom MENU logSettingsMenu=
{
	6,
	{ 
		{ (rom MENU*)0, { (rom char*)"Encoding Error (%)", 			setLogError,		&logError,   	2, 0 } },
		{ (rom MENU*)0, { (rom char*)"Sampling Period (sec.)", 		setLogPeriod, 		&logPeriod,	 	0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Ch.1 Logging", 		setCh1, 			0, 	0, &logChannelString[0][0] } },
		{ (rom MENU*)0, { (rom char*)"Ch.2 Logging", 		setCh2, 			0,	0, &logChannelString[1][0] } },
		{ (rom MENU*)0, { (rom char*)"Ch.3 Logging", 		setCh3, 			0,	0, &logChannelString[2][0] } },
		{ (rom MENU*)0, { (rom char*)"Ch.4 Logging", 		setCh4, 			0,	0, &logChannelString[3][0] } },
	}
};

rom MENU displayMenu=
{
	2,
	{ 
		{ (rom MENU*)0, { (rom char*)"Display Brightness (%)", setBrightnessPercent,		&brightnessPercent, 0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Display Timeout (sec.)", setTimeOutSeconds, 			&timeOutInSeconds, 	0, 0 } },
	}
};

rom MENU mainMenu=
{
	6,
	{ 
		{ (MENU*)&batterySettingsMenu, 		{ (rom char*)"Battery Menu", 0, 0, 0, 0 } },
		{ (MENU*)&calibrationSettingsMenu, 	{ (rom char*)"Calibration Menu", 0, 0, 0, 0 } },
		{ (MENU*)&displayMenu, 				{ (rom char*)"Display Menu", 0, 0, 0, 0 } },
		{ (MENU*)&systemSettingsMenu,  		{ (rom char*)"System Menu", 0, 0, 0, 0 } },
		{ (MENU*)&logSettingsMenu,  		{ (rom char*)"Logging Menu", 0, 0, 0, 0 } },
		{ (MENU*)0,	  						{ (rom char*)"Restore All Defaults", restoreAllDefaults, 0, 0, 0 } }
	}
};

void myISRHigh(void);
void myISR(void);

#pragma code

#pragma code _HIGH_INTERRUPT_VECTOR = 0x000008
void _high_ISR (void)
{
    _asm goto myISRHigh _endasm
	//_asm goto RM_HIGH_INTERRUPT_VECTOR _endasm
}

#pragma code _LOW_INTERRUPT_VECTOR = 0x000018
void _low_ISR (void)
{
    _asm goto myISR _endasm
	//_asm goto RM_LOW_INTERRUPT_VECTOR _endasm
}
//
#pragma code
//
/** D E C L A R A T I O N S **************************************************/
//
//*****************************************************************************
// Keys pressed FIFO functions
//*****************************************************************************
int WriteRLEBuffer(int address, byte data)
{
	// access function for the split buffer in different RAM sections
	// returns the modulo address
	while(address>=RLE_BUFFER_SIZE)
	{
	address-=RLE_BUFFER_SIZE;
	}
	if(address<HALF_RLE_BUFFER_SIZE)
	{
	RLEBuffer[address]=data;
	} else
	{
	RLEBuffer2[address]=data;
	}
	return address;
}

int ReadRLEBuffer(int address, byte* data)
{
	// returns the data in *data and the address as the main return value
	while(address>=RLE_BUFFER_SIZE)
	{
	address-=RLE_BUFFER_SIZE;
	}
	if(address<HALF_RLE_BUFFER_SIZE)
	{
	*data=RLEBuffer[address];
	} else
	{
	*data=RLEBuffer2[address];
	}
	return address;
}

byte getRLE(void)
{
	byte g;
	if(RLEFull>0)
    	{
	    RLEGetPtr=ReadRLEBuffer(RLEGetPtr, &g);
	    RLEGetPtr++;
	    RLEFull--;
	    } else g=0;
	return g;
}

void putRLE(byte k)
{
	if(RLEFull>=RLE_BUFFER_SIZE)
	 {
  	 // the buffer is full, so we overwrite some data to make space for new data!
	 RLEGetPtr++;
	 RLEFull--;
	 }
	 // then write!
	 RLEPtr=WriteRLEBuffer(RLEPtr, k);
	 RLEPtr++;
	 RLEFull++;
}    

void initRLE(void)
{
	byte i;
    RLEGetPtr=0;
    RLEFull=0;   
    RLEPtr=0;
	for(i=0; i<RLE_CHANNELS;i++)
	{ 
	RLEMultiplicity[i]=0;
	RLEData[i]=0.0;
	}
}    

void flushRLEData(byte channel)
{
	byte *ptr;
	byte *ptr2;
	byte x;
	unsigned int ch;

	channel&=RLE_CHANNEL_MASK;
	ch=(unsigned int)channel;
	RLEMultiplicity[channel]|=(ch<<RLE_BIT_SHIFT);
	ptr=(byte*)&RLEMultiplicity[channel];
	ptr2=(byte*)&RLEData[channel];
	putRLE(*ptr++);
	putRLE(*ptr);
	putRLE(*ptr2++);
	putRLE(*ptr2++);
	putRLE(*ptr2++);
	putRLE(*ptr2);
	RLEMultiplicity[channel]=0;
}

void flushAllRLEChannels(void)
{
	byte i;
	for(i=0; i<RLE_CHANNELS; i++)
	{
	flushRLEData(i);
	}
}

void retrieveRLEData(byte *multiplicityptr, byte* doubleptr)
{
	*multiplicityptr++=getRLE();
	*multiplicityptr=getRLE();
	*doubleptr++=getRLE();
	*doubleptr++=getRLE();
	*doubleptr++=getRLE();
	*doubleptr=getRLE();
}

void popRLEData(byte* outptr)
{
	byte i;
	byte* outptr2;
	byte x;

	flushAllRLEChannels();
	outptr2=outptr;
	outptr++;
	i=0;
	while(i<16)
	{
	retrieveRLEData(outptr, outptr+2);
	x=*outptr;
	x|=(*(outptr+1));
	if(x==0)break;
	outptr+=6;	
	i++;
	}
	*outptr2=i;

}

void pushRLEData(byte channel, double data, double percenterror)
{
	double f;
	double error;

	channel&=RLE_CHANNEL_MASK;
	if(RLEMultiplicity[channel]==0)
	{
		RLEMultiplicity[channel]++;
		RLEData[channel]=data;
	} else
	{
		error=RLEData[channel];
		f=(data-error);
		if(f<=0.0)f=-f;
		error*=(percenterror/100.0);
		if(error<=0.0)error=-error;
		if(f<error)
		{
		if(RLEMultiplicity[channel]<MAX_RLE_MULTIPLICITY)RLEMultiplicity[channel]++;
		else
		flushRLEData(channel);
		}
		else
		{
		flushRLEData(channel);
		}
	}
}

// End of RLE Subsystem
void goToLowPowerMode(void)
{
	if(USBSENSE==0)
	{
	OSCCON=0x80;
	_asm
		sleep
	_endasm
	}
}

void delayMs(unsigned int delay)
{
	unsigned int n;
	//
 	while(delay--)
	{
	for(n = 0; n < 500; n++) 
	{
	// the following must be around a 2us delay...	
	_asm
		nop
		nop
		nop
		nop
	_endasm
	}	
 	}  
}

//*****************************************************************************
// Keys pressed FIFO functions
//*****************************************************************************
byte getKey(void)
{
	byte g;
	if(keyFull>0)
    	{
	    g=keyBuffer[keyGetPtr];
	    keyGetPtr++;
	    if(keyGetPtr>=KEY_BUFFER_SIZE)keyGetPtr=0;
	    keyFull--;
	    } else return 0xFF;
	return translateKeyCode(g);
}

byte getNumberKey(void)
{
	byte g;
	if(keyFull>0)
    	{
	    g=keyBuffer[keyGetPtr];
	    keyGetPtr++;
	    if(keyGetPtr>=KEY_BUFFER_SIZE)keyGetPtr=0;
	    keyFull--;
	    } else return 0xFF;
	return translateNumberCode(g);
}

void putKey(byte k)
{
	resetTimeOut();
	if(!standBy)setBrightnessLevel(brightnessPercent);
    if(keyFull<KEY_BUFFER_SIZE)
     {
	 keyBuffer[keyPtr]=k;
	 keyPtr++;
	 if(keyPtr>=KEY_BUFFER_SIZE)keyPtr=0;
	 keyFull++;
	 }
}    

void initKeys(void)
{
    keyGetPtr=0;
    keyFull=0;   
    keyPtr=0;
}    

byte swapBits(byte input)
{
	byte result;
	result=0;
	if(input&0x80)result|=0x10;
	if(input&0x40)result|=0x20;
	if(input&0x20)result|=0x40;
	if(input&0x10)result|=0x80;
	return result;
}

void writeLCD(byte data, byte command)
{
		int i;
		byte interrupt;
		byte firstdata;
		byte seconddata;

		firstdata=swapBits(data);
		seconddata=swapBits(data<<4);
		LCDE=0;
		if(command==LCDCOMMAND)LCDRS=0; else LCDRS=1;
		PORTB=firstdata;
		_asm
			nop
			nop
		_endasm
		interrupt=INTCONbits.GIE;
		INTCONbits.GIE=0;
		LCDE=1;
		_asm
			nop
			nop
		_endasm
		LCDE=0;
		if(interrupt)INTCONbits.GIE=1;
		PORTB=seconddata;
		_asm
			nop
			nop
		_endasm
		interrupt=INTCONbits.GIE;
		INTCONbits.GIE=0;
		LCDE=1;
		_asm
			nop
			nop
		_endasm
		LCDE=0;
		if(interrupt)INTCONbits.GIE=1;
		PORTB&=scanLn;
		for(i=0; i<LCD_DELAY; i++)
		{	
			_asm
				nop
				nop
				nop
				nop
			_endasm
		}
}

void writeScreen(byte* inptr)
{
	byte i;
	writeLCD(0x80, LCDCOMMAND);
	for(i=0; i<(LCD_SIZE/2); i++)
	{
	writeLCD(*inptr, LCDDATA);
	inptr++;
	}
	writeLCD(0xC0, LCDCOMMAND);
	for(i=0; i<(LCD_SIZE/2); i++)
	{
	writeLCD(*inptr, LCDDATA);
	inptr++;
	}
}

void updateScreen(byte *inptr)
{
	if((!standBy)||(BUTTON==0))
	{
		writeScreen(inptr);
	}
}

void updateLCD(void)
{
	// function =  0 then update straight to the LCD screen...
	// function = -1 then update to buffer, then next update will shift left...
	// function = +1 then update to buffer, then next update will shift right... 
	
	int i;
	static int lastCall=0;
	if(lcdTransitionType==0)
	{
		if(lastCall==0)
		{
		updateScreen(&lcdScreen[0]);
		} else
		{
		  for(i=0;i<(LCD_SIZE/2); i++)
		  {	
			updateScreen(&oldlcdScreen[0]);
			if(lastCall==1)
			{
			shiftGeneral(0, 15, lastCall, 2, &oldlcdScreen[0]);
			oldlcdScreen[15]=lcdScreen[0];
			oldlcdScreen[31]=lcdScreen[16];
			shiftGeneral(0, 15, lastCall, 2, &lcdScreen[0]);
			} else
			{
			shiftGeneral(0, 15, lastCall, 2, &oldlcdScreen[0]);
			oldlcdScreen[0]=lcdScreen[15];
			oldlcdScreen[16]=lcdScreen[31];
			shiftGeneral(0, 15, lastCall, 2, &lcdScreen[0]);
			}
		 	delayMs(SCROLL_DELAY/4);
 		  }
			updateScreen(&oldlcdScreen[0]);
			for(i=0; i<LCD_SIZE; i++)lcdScreen[i]=oldlcdScreen[i];
		}
		lcdTransitionType=0;
		lastCall=0;
	} else
	{
		for(i=0; i<LCD_SIZE; i++)oldlcdScreen[i]=lcdScreen[i];
		lastCall=lcdTransitionType;
		lcdTransitionType=0;
	} 
}

void clearLCD(void)
{
	int i;
	for(i=0; i<LCD_SIZE; i++)
	{
	lcdScreen[i]=' ';
	}
	updateLCD();
}

void initLCD(void)
{
		PORTC=0;
		PORTB=0;
		writeLCD(0x28, LCDCOMMAND);
		writeLCD(0x28, LCDCOMMAND);
		delayMs(200);
		writeLCD(0x0C, LCDCOMMAND);
		delayMs(200);
		writeLCD(0x06, LCDCOMMAND);
		delayMs(200);
		clearLCD();
		doBeep(150);				// pause 160ms
}

void shiftGeneral(int from, int to, int offset, int columns, byte* inptr)
{
	int i, j, k;

	j=0;
	k=0;
	while(j<columns)
	{
	if(offset>0)
	{
		i=from; 
		while(i<to)
		{
		*(inptr+i+k)=*(inptr+i+k+offset);
		i++;
		}
		*(inptr+to+k)=' ';
	} else
	if(offset<0)
	{
		i=to;
		while(i>from)
		{
		*(inptr+i+k)=*(inptr+i+k+offset);
		i--;
		}
		*(inptr+from+k)=' ';
	}
	k+=(LCD_SIZE/2);
	j++;
	}
}

void shiftUpLCD(void)
{
	int i;
		
	for(i=0; i<(LCD_SIZE/2); i++)
	{
		lcdScreen[i]=lcdScreen[i+(LCD_SIZE/2)];
		lcdScreen[i+(LCD_SIZE/2)]=' ';
	}
}

void shiftDownLCD(void)
{
	int i;
		
	for(i=0; i<(LCD_SIZE/2); i++)
	{
		lcdScreen[i+(LCD_SIZE/2)]=lcdScreen[i];
		lcdScreen[i]=' ';
	}
}

void shiftLeftLCD(void)
{
	shiftGeneral(0, 15, 1, 2, &lcdScreen[0]);
}

void shiftRightLCD(void)
{
	shiftGeneral(0, 15, -1, 2, &lcdScreen[0]);
}

void leftTransitionLCD(void)
{
	int i, j;
	for(i=0; i<(LCD_SIZE/2); i++)
	{
		shiftLeftLCD();
		updateLCD();
		delayMs(SCROLL_DELAY/4);
	}
}

void rightTransitionLCD(void)
{
	int i, j;
	for(i=0; i<(LCD_SIZE/2); i++)
	{
		shiftRightLCD();
		updateLCD();
		delayMs(SCROLL_DELAY/4);
	}
}

void writeStringLCD(byte cursor, rom char* instr)
{
		while((*instr)!='\0')
		{
			if(cursor<LCD_SIZE)lcdScreen[cursor++]=*instr;
			instr++;
		}
		//updateLCD();
}

void writeStringLCDDirect(byte cursor, rom char* instr)
{
		writeLCD(cursor, LCDCOMMAND);
		while((*instr)!='\0')
		{
			writeLCD(*instr, LCDDATA);
			instr++;
		}
}

void writeStringRamLCD(byte cursor, ram char* instr, int padding)
{
		while((*instr)!='\0')
		{
			if(cursor<LCD_SIZE)lcdScreen[cursor++]=*instr;
			instr++;
			padding--;
		}
		while(padding>0)
		{
			if(cursor<LCD_SIZE)lcdScreen[cursor++]=' ';
			padding--;
		}
		//updateLCD();
}

byte disFix(byte in)
{
	in&=0x0F;
	if(in>9)in+=7;
	return in+0x30;
}

int printITOA(unsigned long xx, int blanking, int decimal, int mode)
{
		// unsigned integers from 0 to 65535
		// blanking=0 implies zero blanking
		// decimal=   number of digits before the decimal to be printed if 0 then do not print a decimal point (blanking stops after the decimal)
		int i, j;
		unsigned long ux;
		char c;
		
		ux=(unsigned long)xx;
		j=NUM_DIGITS+1;
		for(i=NUM_DIGITS; i>=0; i--)
		{
		c=(char)(0x30+(0x0F&(ux%10)));
		ux=ux/10;
		if(j==decimal){ buffer[j--]='.'; }
		buffer[j--]=c;
		}
		buffer[NUM_DIGITS+2]=0;
		j=j+1;
		if(blanking==0)while((buffer[j]==0x30)&&(j<NUM_DIGITS))j++;
		if(mode==0)if((j>0)&&(buffer[j]=='.'))j--;
		return j;
}

rom double MultiplierF[10]={ 1.0, 10.0, 100.0, 1000.0, 10000.0, 100000.0, 1000000.0, 10000000.0, 100000000.0, 1000000000.0 };

int printFDecimal(double f, int numdec, int mode)
{
		if(f<0.0)f=-f;
		if(numdec<0)numdec=0; else if(numdec>10)numdec=10;
		f*=MultiplierF[numdec];
		if(roundingMode!=0)f+=0.5;
		return printITOA(f, 0, (NUM_DIGITS+1)-numdec, mode);
}

int printFSignedDecimal(double f, int numdec, int mode)
{
		byte k;
		char c;

		if(numdec<0)numdec=0; else if(numdec>10)numdec=10;
		if(f<0){ f=-f; c='-'; } else c='+';
		f*=MultiplierF[numdec];
		if(roundingMode!=0)f+=0.5;
		k=printITOA(f, 0, (NUM_DIGITS+1)-numdec, mode);
		k--;
		buffer[k]=c;
		return k;
}

void disFLCD(byte cursor, double f, int numdec, int mode, int padding)
{
	int i;

	if(f<0)
	{
	f=-f;
	if(cursor<LCD_SIZE)lcdScreen[cursor]='-';
	} else
	{
	if(cursor<LCD_SIZE)lcdScreen[cursor]='+';
	}
	i=printFDecimal(f, numdec, mode);
	writeStringRamLCD(cursor+1, (ram char*)&buffer[i], padding);
	updateLCD();
}

void disFUnsignedLCD(byte cursor, double f, int numdec, int mode, int padding)
{
	int i;

	if(f<0)f=-f;
	i=printFDecimal(f, numdec, mode);
	writeStringRamLCD(cursor+1, (ram char*)&buffer[i], padding);
	updateLCD();
}

/*
void disaLCD(byte cursor, byte a)
{
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a>>4);
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a);
		updateLCD();
}
*/
/*
void diswordLCD(byte cursor, unsigned int a)
{
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a>>12);
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a>>8);
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a>>4);
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a);
		updateLCD();
}
*/

void initPowerSystem(void)
{
	TRISAbits.TRISA4=0;		// make it an output!
	POWERON=0;				// turn power on!
}

byte usbSense(void)
{
	if(USBSENSE)
	{
		yesNoUSBString[0]='Y';
		yesNoUSBString[1]='e';
		yesNoUSBString[2]='s';
		yesNoUSBString[3]='\0';
		return 1;
	} else
	{
		USBConnect();
		yesNoUSBString[0]='N';
		yesNoUSBString[1]='o';
		yesNoUSBString[2]='\0';
		return 0;
	}
}

void powerOff(void)
{
	if(USBSENSE==0)
	{
		POWERON=1;
		OSCCON=0;
		_asm
			sleep
		_endasm
	}
}

void goOutOfStandBy(void)
{
	PIE2bits.TMR3IE=1;
	standBy=0;
}

void goInToStandBy(void)
{
	PIE2bits.TMR3IE=0;
	standBy=1;
}

void checkStandBy(void)
{
	double f, g, h;

	// a small amount of hysteresis is factored into the going in and out of standby mode...
	if(enableStandByDetect)
	{
	 f=chargeCurrent;
	 if(f<=0.0)f=-f;
	 g=loadCurrent;
	 if(g<=0.0)g=-g;
	 if(standBy)
	 {
		h=currentThreshold*1.5;
		if((BUTTON==0)||(USBSENSE)||(f>=h)||(g>=h))
		{
			goOutOfStandBy();
		}
	 } else
	 {
		if((USBSENSE==0)&&(f<=currentThreshold)&&(g<=currentThreshold))
		{
			goInToStandBy();
		}
	 }
	} else standBy=0;
}

rom byte keyCodes[16]={ 'A', '3', '2', '1', 'B', '6', '5', '4', 'C', '9', '8', '7', 'D', '#', '0', '*' };
rom byte numberCodes[16]={ '+', '3', '2', '1', '-', '6', '5', '4', 'C', '9', '8', '7', 'D', '#', '0', '.' };

byte translateKeyCode(byte rawCode)
{
	return keyCodes[(rawCode & 0x0F)];
}

byte translateNumberCode(byte rawCode)
{
	return numberCodes[(rawCode & 0x0F)];
}

double enterNumber(byte cursor, int numdec, int maxchars, double defaultValue)
{
	double f;
	ram char string[16];
	if(maxchars>15)maxchars=15;
	enterNumberString(cursor, &string[0], maxchars);
	if(string[0]=='\0')
	{
	f=defaultValue;
	}
	else
	{
	f=atof(&string[0]);
	}
	disFLCD(cursor, f, numdec, 0, maxchars);
	return f;
}

void enterNumberString(byte cursor, ram char* string, int maxchars)
{
	byte key, index, i;

	key=0;
	index=0;
  	while(key!='#')
	{
	while(keyFull==0);
	key=getNumberKey();
		if((key!='D')&&(key!='#'))
		{
			if(index<maxchars)
			{
			lcdScreen[index+cursor]=key;  
			updateLCD();
			string[index]=key;
			}
			if(index<maxchars)index++;
		} else
		if(key=='D')
		{ 
			if(index>0)index--; 
			i=index;
			while(i<maxchars)
			{
			lcdScreen[i+cursor]=' ';
			updateLCD();
			string[i]=' ';
			i++;
			}
		} else
		if(key=='#')
		{
			string[index]='\0';
		}
	}
}

void initKeypad(void)
{
	scanCounter=0;
	scanCode=0;
	scanLn=0;
	T3CON=0;				// turn off
	TMR3H=0;
	TMR3L=1;
	T3CON=0b10100001;		// 1:8 prescaler from system clock synchronised , 16 bit operation
	PIR2bits.TMR3IF=0;
	IPR2bits.TMR3IP=0;		// low priority interrupt
	PIE2bits.TMR3IE=1;		// enable interrupt
	INTCON2bits.RBPU=0;		// enable weak pull ups on port B
//	INTCONbits.RBIF=0;
//	INTCON2bits.RBIP=0;		// low priority interrupt
//	INTCONbits.RBIE=0;		// enable interrupt on change interrupt
}

void initSampling(void)
{
	// initialize the sampling system
	// which samples the 5 ADC readings
	// periodically in the background
	int i;

	samplingIndex=0;
	samplingCount=0;
	for(i=0; i<(NUM_SAMPLES-2); i++)
	{
	//sample[i].lastSampled=0;
	sample[i].value=0.0;
	sample[i].avgValue=0.0;	
	//sample[i].delta=0.0;
	sample[i].updated=0;
	sample[i].voltageDivider=1.0;
	}	
	capacityAs=0.0;
	deltaCapacityAs=0.0;
	time=0.0;
	T1CON=0;				// turn off
	TMR1H=0;
	TMR1L=1;				
	T1CON=0x81;				// 1:1 prescaler from system clock synchronised, 16 bit operation
	PIR1bits.TMR1IF=0;
	IPR1bits.TMR1IP=1;		// high interrupt priority
	PIE1bits.TMR1IE=1;		// enable interrupt
}

int initADC(void)
{
	ADCON1=0x0A;			// enable AN0 to AN4 as analog inputs the rest as digital pins
	ADCON2=0xBD;			// right justified, FOSC/16 (should not be lower than this, ie faster) and 20 TAD
	ADCON0=0x01;
}

unsigned int readAN(byte channel)
{
	unsigned int i;
	// returns a number from 0.0 to 1023.0 (or a bit higher depending on ref voltage)
	TRISCbits.TRISC2=1;		// disable backlight temporarily
	ADCON0&=0xC3;
	i=0x3C & ((channel & 0x0F)<<2);
	ADCON0|=i;				// set the channel
	i=0;
	ADCON0|=0x02;
	while((ADCON0 & 0x02)!=0);
	i=(unsigned int)((((unsigned int)ADRESH)<<8) + ADRESL);
	TRISCbits.TRISC2=0;
	return i;
}

void initDelayTimer(void)
{
	timerDelay=0;
	INTCONbits.TMR0IF=0;
	INTCON2bits.TMR0IP=0;	// interrupt priority
	T0CON=0x87;				// 1:256 prescaler from instruction clock at 12000000Hz, gives around 1400ms ticks
	INTCONbits.TMR0IE=1;
}

void initPWM(void)
{
	// initialize the PWN on CCP1
	CCP1CON=0x00;
	TMR2=0x00;
	PR2=0xFF;
	PIR1bits.TMR2IF=0;			// clear ack
	PIE1bits.TMR2IE=0;			// do not enable interrupt
	IPR1bits.TMR2IP=0;			// low priority
	T2CON=0x7F;					
	TRISCbits.TRISC2=0;
	CCPR1L=0;
	CCPR1H=0;
	TMR2=0xFE;					// note that CCP1RH is not latched in unitl there's a match between TMR2 and PR2 ie the cycle starts again.
	PIR1bits.CCP1IF=0;
	PIE1bits.CCP1IE=0;
	IPR1bits.CCP1IP=0;
	CCP1CON=0x0C;				// enable PWM
}

int putPWM(byte xx)
{
	// only 8 bits are used of xx 
	brightness=(0xFE & xx);
	CCPR1L=(brightness);
}

void USBConnect(void)
{
	mInitializeUSBDriver();     // See usbdrv.h
 	USBCheckBusStatus();        // Modified to always enable USB module
}

void initBeep(void)
{
	BEEP=0;
}

void doBeep(int delay)
{
	if(beepOn)
	{
		alarmOn|=2;
		BEEP=1;
		delayMs(delay);
		BEEP=0;
		alarmOn&=~2;
	} else BEEP=0;
}

void idF(int i)
{

}

void setAlarmInternal(int i)
{
	if(alarmCapacity<0.0)alarmCapacity=0.0; else if(alarmCapacity>100.0)alarmCapacity=100.0;
}


ram char* copyString(ram char* instr, ram char* outstr, char delimiter)
{
	while((*instr)!=delimiter)
	{
	*outstr++=*instr++;
	}
	return outstr;
}

void updateTimeString(double floatminutes)
{
	// update the time string with minutes in double...
	ram char* outstr;
	long days;
	int hours;
	int minutes;
	byte k;

	if(floatminutes<0.0)floatminutes=0.0;
	days=(long)(floatminutes/1440.0);
	floatminutes-=(days*1440.0);
	hours=(int)(floatminutes/60.0);
	floatminutes-=(hours*60.0);
	minutes=(int)floatminutes;
	outstr=&timeString[0];
	k=printFDecimal(days, 0, 0);
	outstr=copyString(&buffer[k], outstr, '.');
	*outstr++='d';
	*outstr++=' ';
	k=printFDecimal(hours,0,0);
	outstr=copyString(&buffer[k], outstr, '.');
	*outstr++='h';
	*outstr++=' ';
	k=printFDecimal(minutes,0,0);
	outstr=copyString(&buffer[k], outstr, '.');
	*outstr++='m';
	*outstr='\0';
}

void clearWithoutUpdate(void)
{
	int i;
	for(i=0; i<32; i++)lcdScreen[i]=' ';
}

void clearBottomLine(void)
{
	int i;
	for(i=16; i<32; i++)lcdScreen[i]=' ';
	updateLCD();
}

void flushAndWaitForKey(void)
{
	initKeys();
	while(keyFull==0);
}

void writeTitleStringMenu(rom char* instr, double min, double max, byte numdec)
{
		clearBottomLine();
		disFUnsignedLCD(16, min, numdec, 0, 5);
		lcdScreen[22]='-';
		disFUnsignedLCD(23, max, numdec, 0, 5);
		writeStringLCD(0, instr);
		updateLCD();
}

void doEnterNumberMenu(rom char* instr, double min, double max, byte numdec, double* result)
{
	byte i;
	i=0;
	while((i==0)||((*result)>max)||((*result)<min))
	{
	i=1;
	writeTitleStringMenu(instr, min, max, numdec);
	flushAndWaitForKey();
	clearBottomLine();
	(*result)=enterNumber(16, numdec, 16, *result);
	}
	delayMs(1000);
	clearLCD();
	writeStringLCD(0, (rom char*)"Entered Value:  ");
	disFLCD(16, *result, numdec, 0, 16);
	delayMs(1000);
}

void printPercentageBar(byte from, byte to, double percentage)
{
	byte i, threshold;
	double f;

	f=percentage/100.0;
	i=from;
	threshold=(byte)((to+1)*f+((from)*(1.0-f)));
	while(i<=to)
	{
	if(i<threshold)lcdScreen[i]=0xFF ; else lcdScreen[i]=' ';
	i++;
	}
}

void doEnterValueUpAndDownMenu(rom char* instr, double* value, byte numdec, void (*setFunction)(double), double step)
{
	char keyPressed;

	clearLCD();
	writeStringLCD(0, instr);
	updateLCD();
	keyPressed=0;
	while(keyPressed!='#')
	{
	setFunction(*value);
	disFUnsignedLCD(27, *value, numdec, 0, 5);
	lcdScreen[16]='[';
	lcdScreen[26]=']';
	printPercentageBar(17, 25, *value);
	updateLCD();
		while(keyFull==0);
		keyPressed=getKey();
		switch(keyPressed)
		{
			case 'A':
				*value+=step;
				break;
			case 'B':
				*value-=step;
				break;
			default:
				break;
		}
	}
}

void doEnterOptionsMenu(rom char* instr, byte* valueIndex, ram char* displayString, void (*setFunction)(byte), byte modulus)
{
	char keyPressed;

	clearLCD();
	writeStringLCD(0, instr);
	updateLCD();
	keyPressed=0;
	while(keyPressed!='#')
	{
	setFunction(*valueIndex);
	lcdScreen[16]=(char)(0x31+(*valueIndex));
	lcdScreen[17]=':';
	writeStringRamLCD(19, displayString, 13);
	updateLCD();
		while(keyFull==0);
		keyPressed=getKey();
		switch(keyPressed)
		{
			case 'A':
				if((*valueIndex)>0)(*valueIndex)--;
				break;
			case 'B':
				if((*valueIndex)<(modulus-1))(*valueIndex)++;
				break;
			default:
				break;
		}
	}
}

void displaySettingMenu(rom char* instr, byte numdec, double* result, byte signedDisplay, ram char* valueString, int blocking)
{
	byte key;
	if(blocking==1)initKeys();				// flush queue
	key=0;
	while(key!='#')
	{
	clearWithoutUpdate();
	writeStringLCD(0, instr);
	if(valueString==0)
	{
	if(signedDisplay)disFLCD(16, *result, numdec, 0, 16); else disFUnsignedLCD(16, *result, numdec, 0, 16);
	} else
	{
	writeStringRamLCD(17, valueString, 15);
	updateLCD();
	}
		if(blocking==1)
		{
		delayMs(SCROLL_DELAY*2);
		key=getKey();
		initKeys();
		} 
		else 
			break;
	}
}

int romStringLength(rom char *instr)
{
	// returns the length of the string in rom
	int i;
	i=0;
	while((*instr)!='\0')
	{
	i++;
	instr++;
	}
	return i;
}

void displayTwoSettingsMenu(rom char* instr1, byte numdec1, double* result1, byte signedDisplay1, ram char* valueString1, rom char* instr2, byte numdec2, double* result2, byte signedDisplay2, ram char* valueString2, int blocking)
{
	byte key, offset;
	if(blocking==1)initKeys();				// flush queue
	key=0;
	while(key!='#')
	{
		clearWithoutUpdate();
		writeStringLCD(0, instr1);
		offset=romStringLength(instr1);
		if(valueString1==0)
		{
			if(signedDisplay1)disFLCD(offset, *result1, numdec1, 0, 16); else disFUnsignedLCD(offset, *result1, numdec1, 0, 16);
		} 
		else
		{
			writeStringRamLCD(offset, valueString1, 16);
			updateLCD();
		}

		writeStringLCD(16, instr2);
		offset=romStringLength(instr2);
		if(valueString2==0)
		{
			if(signedDisplay1)disFLCD(16+offset, *result2, numdec2, 0, 16); else disFUnsignedLCD(offset+16, *result2, numdec2, 0, 16);
		} 
		else
		{
			writeStringRamLCD(offset+16, valueString2, 16);
			updateLCD();
		}
		if(blocking==1)
		{
		delayMs(SCROLL_DELAY*2);
		key=getKey();
		initKeys();
		} 
		else 
			break;
	}
}

void seeBatteryChargeAndMinutes(int i)
{
	updateTimeString(batteryMinutes);
	displayTwoSettingsMenu(	(rom char*)"Chrg(%): ", 0, &batteryPercent, 1, 0,
							(rom char*)"Time: ", 0, 0, 0, &timeString[0],
							i);
}

void seeBatteryVoltageCurrent(int i)
{
	displayTwoSettingsMenu(	(rom char*)"Volts: ", 1, &batteryVoltage, 1, 0,
							(rom char*)"Amps : ", 3, &netCurrent, 1, 0,
							i);
}

void seeChargeCurrentAndLoadCurrent(int i)
{
	displayTwoSettingsMenu(	(rom char*)"Chrg(A): ", 3, &chargeCurrent, 1, 0,
							(rom char*)"Load(A): ", 3, &loadCurrent, 1, 0,
							i);
}

void seeCapacityDouble(int i)
{
	displayTwoSettingsMenu(	(rom char*)"Cty.(AH):", 0, &batteryCapacityAH, 1, 0,
							(rom char*)"Load(A): ", 3, &loadCurrent, 1, 0,
							i);
}

void seeCircuit(int i)
{
	displayTwoSettingsMenu(	(rom char*)"Circt.(mA): ", 0, &circuitCurrent, 1, 0,
							(rom char*)"Circt.(V): ", 1, &circuitVoltage, 1, 0,
							i);
}

void seeVoltages(int i)
{
	displayTwoSettingsMenu(	(rom char*)"Volts (V): ", 1, &batteryVoltage, 1, 0,
							(rom char*)"Circt.(V): ", 1, &circuitVoltage, 1, 0,
							i);
}

/*
void seeAllScrolling(int i)
{
	clearWithoutUpdate();
	writeStringLCD(0, (rom char*)"All Scrolling");
	updateLCD();
}
*/

void setDividerLow(int i)
{
	doEnterNumberMenu((rom char*) "Lower Volt. Now?", 0.0, 60.0 , 2, &dividerLow);
}

void setCutOffVoltage(int i)
{
	doEnterNumberMenu((rom char*) "Cut Off Voltage?", 0.0, 60.0 , 1, &cutOffVoltage);
}

void setBrightnessPercent(int i)
{
	doEnterValueUpAndDownMenu((rom char*)"Brightness (%)? ", &brightnessPercent, 0, setBrightnessLevel, 5.0);
}

void setAlarm(int i)
{
	doEnterValueUpAndDownMenu((rom char*)"Alarm Cty. (%)? ", &alarmCapacity, 0, setAlarmInternal, 1.0);
}

void setDividerHigh(int i)
{
	doEnterNumberMenu((rom char*) "Higher Volt.Now?", 0.0, 60.0 , 2, &dividerHigh);
}

void setThreshold(int i)
{
	doEnterNumberMenu((rom char*) "Threshold (A)?", 0.001, 10.0, 3, &currentThreshold);
}

void setVoltageRail50(int i)
{
	doEnterNumberMenu((rom char*) "+5.0V Rail (V)? ", 4.0, 6.0, 2, &voltageRail50);
}

void setVoltageRail33(int i)
{
	doEnterNumberMenu((rom char*) "+3.3V Rail (V)? ", 3.0, 3.6, 2, &voltageRail33);
}

void setFullCapacity(int i)
{	
	doEnterNumberMenu((rom char*) "Capacity (AH)?  ", 0.0, 999999.0, 1, &fullCapacity);
}

void setDifferentialGain(int i)
{	
	doEnterNumberMenu((rom char*) "Differentl.Gain?", 1.0, 1000.0, 2, &differentialGain);
}

void setSenseResistance(int i)
{	
	doEnterNumberMenu((rom char*) "Sense Res.(\xF4)?", 0.1, 100.0, 1, &senseResistance);
}

void setShuntResistance(int i)
{	
	doEnterNumberMenu((rom char*) "Shunt Res.(m\xF4)?", 1.0, 1000.0, 0, &shuntResistance);
}

void setChargingEfficiency(int i)
{	
	doEnterNumberMenu((rom char*) "Chrg.Efficiency?", 0.1, 100.0, 1, &chargingEfficiency);
}

void seeFirmwareVersion(int i)
{
	displaySettingMenu((rom char*)"Firmware Version", 2, &firmwareVersion, 0, 0, i);
}

void seeBatteryPercent(int i)
{
	displaySettingMenu((rom char*)"Battery Chrg.(%)", 1, &batteryPercent, 1, 0, i);	// signed
}

void seeBatteryCapacityAH(int i)
{
	displaySettingMenu((rom char*)"Batt.Capcty.(AH)", 3, &batteryCapacityAH, 1, 0, i); // signed
}

void seeBatteryVoltage(int i)
{
	displaySettingMenu((rom char*)"Batt. Voltage(V)", 1, &batteryVoltage, 0, 0, i);
}

void seeVoltageRail50(int i)
{
	displaySettingMenu((rom char*)"+5.0V Rail (V)  ", 2, &voltageRail50, 0, 0, i);
}

void seeVUSBRail(int i)
{
	displaySettingMenu((rom char*)"+3.3V Rail (V)  ", 2, &vusbRail, 0, 0, i);
}

void seeCircuitVoltage(int i)
{
	displaySettingMenu((rom char*)"Circuit Volt.(V)", 1, &circuitVoltage, 0, 0, i);
}

void seeLoadDiffAmpVoltage(int i)
{
	displaySettingMenu((rom char*)"Load Amp (V)    ", 3, &loadDiffAmpVoltage, 0, 0, i);
}

void seeChargeDiffAmpVoltage(int i)
{
	displaySettingMenu((rom char*)"Charge Amp (V)  ", 3, &chargeDiffAmpVoltage, 0, 0, i);
}

void seeLoadCurrent(int i)
{
	displaySettingMenu((rom char*)"Load Current(A) ", 3, &loadCurrent, 0, 0, i);
}

void seeChargeCurrent(int i)
{
	displaySettingMenu((rom char*)"Chrg.Current(A) ", 3, &chargeCurrent, 0, 0, i);
}

void seeCircuitCurrent(int i)
{
	displaySettingMenu((rom char*)"Circuit Draw(mA)", 0, &circuitCurrent, 0, 0, i);
}

void seeNetBatteryDrain(int i)
{
	displaySettingMenu((rom char*)"Net Current (A) ", 3, &netCurrent, 1, 0, i);	// signed
}

void seeBatteryTimeMinutes(int i)
{
	updateTimeString(batteryMinutes);
	displaySettingMenu((rom char*)"Time Remaining  ", 0, 0, 0, &timeString[0], i);
}

void seeUSBStatus(int i)
{
	displaySettingMenu((rom char*)"USB Connected:  ", 0, 0, 0, &yesNoUSBString[0], i);
}

void updateBeeperString(byte x)
{
	beepOn=(x & 1);
	if(beepOn)
	{
		yesNoBeeperString[0]='O';
		yesNoBeeperString[1]='n';
		yesNoBeeperString[2]='\0';
	}
	else
	{
		yesNoBeeperString[0]='O';
		yesNoBeeperString[1]='f';
		yesNoBeeperString[2]='f';
		yesNoBeeperString[3]='\0';
	}
}

void setBeeperStatus(int i)
{
	doEnterOptionsMenu((rom char*)"Beeper Status? ", &beepOn, &yesNoBeeperString[0], &updateBeeperString, 2);
}

void setPeukerts(int i)
{	
	doEnterNumberMenu((rom char*) "Peukert's Const?", 1.0, 1.5, 3, &peukertsConstant);
}

void setLogError(int i)
{	
	doEnterNumberMenu((rom char*) "Log Error (%)?", 0.1, 100.0, 2, &logError);
}

void setCh1(int i)
{
	doEnterOptionsMenu((rom char*) "Ch.1 Logs? ", &logChannel[0], &logChannelString[0][0], &updateLogChannelStrings, LOG_TOTAL+1);
}

void setCh2(int i)
{
	doEnterOptionsMenu((rom char*) "Ch.2 Logs? ", &logChannel[1], &logChannelString[1][0], &updateLogChannelStrings, LOG_TOTAL+1);
}

void setCh3(int i)
{
	doEnterOptionsMenu((rom char*) "Ch.3 Logs? ", &logChannel[2], &logChannelString[2][0], &updateLogChannelStrings, LOG_TOTAL+1);
}

void setCh4(int i)
{
	doEnterOptionsMenu((rom char*) "Ch.4 Logs? ", &logChannel[3], &logChannelString[3][0], &updateLogChannelStrings, LOG_TOTAL+1);
}

void setLogPeriod(int i)
{
	doEnterNumberMenu((rom char*) "Log Period(sec)?", 1.0, 86400.0, 3, &logPeriod);
}

void setChemistry(int i)
{
	doEnterOptionsMenu((rom char*) "Cell Chemistry? ", &cellType, &cellTypeString[0], &writeCellTypeString, 2); 
}

void setMinVoltage(int i)
{
	doEnterNumberMenu((rom char*) "Min. Voltage(V)?", 0.0, 60.0, 1, &minVoltage);
}

void setMaxVoltage(int i)
{
	doEnterNumberMenu((rom char*) "Max. Voltage(V)?", 0.0, 60.0, 1, &maxVoltage);
}

void resetTimeOut(void)
{
	timeOutSecondsCounter=timeOutInSeconds;
}

void setTimeOutSeconds(int i)
{
	doEnterNumberMenu((rom char*) "Time Out (sec.)?", 0.0, 86400.0, 0, &timeOutInSeconds);
	resetTimeOut();
}

void copyMenuString(byte menuIndex, rom MENU* myMenu, char* outString, byte shiftIndex, byte totalLength)
{
	byte i, j, k;
	char n;
	char *str;
	char c;
	rom char* title;

	if(myMenu->menu[menuIndex].body.function!=0)c='['; else c='<';
	*outString++=c;
	*outString++=0x31+menuIndex;
	*outString++=':';
	i=0;
	title=myMenu->menu[menuIndex].body.title;
	while((*title)!='\0')tempbuffer[i++]=*title++;
		// i now points to the end point and the string is copied...	 
		 if(myMenu->menu[menuIndex].body.setting!=0)
		 {
		 // if not a null pointer, then...
		 tempbuffer[i++]=':';
		 tempbuffer[i++]=' ';
		 n=myMenu->menu[menuIndex].body.numdec;
		 if(n<0)n=-n;
		 k=printFSignedDecimal(*(myMenu->menu[menuIndex].body.setting), n, 0);
		 if((myMenu->menu[menuIndex].body.numdec)<0)k++;
		 while(buffer[k]!='\0')tempbuffer[i++]=buffer[k++];
		 } else
		 if(myMenu->menu[menuIndex].body.string!=0)
		 {
		 // a string type
		 tempbuffer[i++]=':';
		 tempbuffer[i++]=' ';
		 str=myMenu->menu[menuIndex].body.string;
		 while(*str!='\0')tempbuffer[i++]=*str++;
		 }
	while(i<MENU_STRING_LENGTH+12)tempbuffer[i++]=' ';
	i=3;
	while(i<(totalLength-1))
			{
			j=i+shiftIndex-3;
			while(j>=(MENU_STRING_LENGTH+12))j-=(MENU_STRING_LENGTH+12);
			*outString++=tempbuffer[j];
			i++;
			}
	if(myMenu->menu[menuIndex].body.function!=0)c=']'; else c='>';
	*outString++=c;
}

void printMenuAt(byte cursor, byte menuIndex, rom MENU* myMenu, byte shiftIndex, byte totalLength)
{
	copyMenuString(menuIndex, myMenu, (char*)&lcdScreen[cursor], shiftIndex, totalLength);
}

void printMenu(byte menuIndex, rom MENU* myMenu, byte shiftIndex)
{
	printMenuAt(0, menuIndex, myMenu, shiftIndex, 16);
	printMenuAt(16, menuIndex+1, myMenu, shiftIndex, 16);
	updateLCD();
}

//*****************************************************************************
// Menu Stack
//*****************************************************************************
rom MENU* popMenu(byte* startIndex)
{
	rom MENU* g;
	if(menuFull>0)
    	{
	    menuPtr--;
		menuFull--;
		g=menuStack[menuPtr];
		(*startIndex)=menuStartIndexStack[menuPtr];
	    } else 
		{
		g=0;
		(*startIndex)=0;
		}
	return g;
}

void pushMenu(rom MENU* k, byte startIndex)
{
    if(menuFull<MENU_STACK_SIZE)
     {
	 menuStack[menuPtr]=k;
	 menuStartIndexStack[menuPtr]=startIndex;
	 menuPtr++;
	 menuFull++;
	 }    
}    

void initMenuStack(void)
{
    menuFull=0;   
    menuPtr=0;
}    

rom MENU* executeMenu(byte startIndex, rom MENU* myMenu)
{
	byte shiftIndex;
	byte exit;
	byte i, j;
	byte cursor;
	byte keyPressed;
	int length;
	char buffer[MENU_STRING_LENGTH+4];
	rom MENU* returnPointer;
	
	initKeys();			// flush any keys in the queue
	shiftIndex=0;
	exit=0;	
	returnPointer=0;
	while(!exit)
	{
		printMenu(startIndex, myMenu, shiftIndex);
		shiftIndex++;
		if(shiftIndex>MENU_STRING_LENGTH)shiftIndex=0;
		if(keyFull>0)
		{
			keyPressed=getKey();
			switch(keyPressed)
			{
				case '0':
					break;
				
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					i=keyPressed-0x31;
					j=i-startIndex;
					if((j==0)||(j==1))exit=1;
					keyPressed=i;
					break;
				
				case 'A':
					if(startIndex>0)startIndex--;
					shiftIndex=0;
					break;

				case 'D':
				case 'C':
					shiftIndex=0;
					break;
				
				case 'B':
					if(startIndex<(myMenu->numMenus-2))startIndex++;
					shiftIndex=0;
					break;
				
				case '*':
					break;
				
				case '#':
					exit=2;
					keyPressed=startIndex;
					lcdTransitionType=-1;
					updateLCD();
					break;
			}
		} else delayMs(SCROLL_DELAY/2);
	}

	if(exit==1)
	{
	
	// if a leaf, execute the command
	if(myMenu->menu[keyPressed].body.function!=0)
		{
		lcdTransitionType=1;
		updateLCD();
		clearLCD();
		myMenu->menu[keyPressed].body.function(1);		// 1= blocking!
		lcdTransitionType=-1;
		updateLCD();
		pushMenu(myMenu, startIndex);
		}
		else 
		{
		returnPointer=(rom MENU*)myMenu->menu[keyPressed].nextItem;
		}
	} 
	return returnPointer;
}

void executeMenuStack(void)
{
	rom MENU* x;
	rom MENU* next;
	byte startIndex;
	byte i;
	
	x=popMenu(&startIndex);
	saveSettings();
	if(x!=0)
	{
		next=executeMenu(startIndex, x);
		if(next==0)executeMenuStack();
	    else
		{
		lcdTransitionType=1;
		updateLCD();
		pushMenu(x, startIndex);
		pushMenu(next, 0);
		executeMenuStack();
		}
	}
}

void writeCellTypeString(byte cType)
{
	if(cType==0)
	{
	cellType=0;
	cellTypeString[0]='N';
	cellTypeString[1]='i';
	cellTypeString[2]='c';
	cellTypeString[3]='k';
	cellTypeString[4]='e';
	cellTypeString[5]='l';
	cellTypeString[6]='\0';
	} else
	{
	cellType=1;
	cellTypeString[0]='L';
	cellTypeString[1]='e';
	cellTypeString[2]='a';
	cellTypeString[3]='d';
	cellTypeString[4]=' ';
	cellTypeString[5]='A';
	cellTypeString[6]='c';
	cellTypeString[7]='i';
	cellTypeString[8]='d';
	cellTypeString[9]='\0';
	}
}

void setBrightnessLevel(double percent)
{
	if(percent<0.0)percent=0.0; else if(percent>100.0)percent=100.0;
	brightnessPercent=percent;
	setBrightness=0xFE & ((byte)(2.55*percent));
}

void initValues(void)
{
	alarmOn=0;
	standBy=0;

	enableStandByDetect=0;
	lcdTransitionType=0;
	firmwareVersion=VERSION;
	logPeriodCounter=0.0;
	batteryPercent=0.0;;			// battery percent charge
	batteryCapacityAH=0.0;;			// battery capacity in AH (Amp Hours)
	batteryVoltage=0.0;				// in Volts
	loadCurrent=0.0;				// in Amps
	circuitCurrent=0.0;				// in mAmps
	netCurrent=0.0;					// net current drain in Amps
	batteryMinutes=0.0;				// remaining Capacity
	chargeCurrent=0.0;				// charging current in A
	loadDiffAmpVoltage=0.0;
	chargeDiffAmpVoltage=0.0;
	circuitVoltage=0.0;
	brightness=0;
}

void initDefaults(void)
{
	updateBeeperString(1);
	logChannel[0]=0;
	logChannel[1]=0;
	logChannel[2]=0;
	logChannel[3]=0;
	cutOffVoltage=0.0;
	alarmCapacity=0.0;
	differentialGain=DEFAULT_DIFFERENTIAL_GAIN;
	shuntResistance=DEFAULT_SHUNT_RESISTANCE;
	senseResistance=DEFAULT_SENSE_RESISTANCE;
	voltageRail50=DEFAULT_VCC;
	voltageRail33=DEFAULT_VUSB;
	dividerLow=DEFAULT_LOW_DIVIDER;
	dividerHigh=DEFAULT_HIGH_DIVIDER;
	chargingEfficiency=DEFAULT_CHARGING_EFFICIENCY;
	displayMode=MODE_DEFAULT;
	roundingMode=DEFAULT_ROUNDINGMODE;
	logError=DEFAULT_LOG_ERROR;
	logPeriod=DEFAULT_LOG_PERIOD;
	brightnessPercent=(DEFAULT_BRIGHTNESS_PERCENT);
	timeOutInSeconds=DEFAULT_TIMEOUT_IN_SECONDS;
	fullCapacity=DEFAULT_FULLCAPACITY;
	peukertsConstant=DEFAULT_PEUKERTS;
	minVoltage=DEFAULT_MINVOLTAGE;
	maxVoltage=DEFAULT_MAXVOLTAGE;
	currentThreshold=DEFAULT_THRESHOLD;
	writeCellTypeString(DEFAULT_CHEMISTRY);
}

void updateSettings(void)
{
	resetTimeOut();
	sample[ANCH_VBATTNEG].voltageDivider=dividerLow;
	sample[ANCH_VBATTPOS].voltageDivider=dividerHigh;
	setBrightnessLevel(brightnessPercent);
	updateBeeperString(beepOn);
	writeCellTypeString(cellType);
	updateLogChannelStrings(0);
}

byte waitForYesOrNoResponse(void)
{
	byte key;
	
	writeStringLCD(16, (rom char*)"*: Yes   #: No  ");
	updateLCD();
	key=0;
	while((key!='*')&&(key!='#'))
	{
		flushAndWaitForKey();
		key=getKey();
	}
	if(key=='*')
	{
		writeStringLCD(24, (rom char*)"        ");
		key=1;
		doBeep(1000);
	} else
	{
		writeStringLCD(16, (rom char*)"        ");
		key=0;
		doBeep(350);
		delayMs(300);
		doBeep(350);
	}
	updateLCD();
	return key;
}

byte areYouSure(void)
{
	clearLCD();
	writeStringLCD(0, (rom char*)"Are You Sure?");
	return waitForYesOrNoResponse();
}

void restoreAllDefaults(void)
{
	if(areYouSure())
	{	
	initDefaults();
	saveSettings();
	WriteEEPROM(0, 0x76);
	}
}

void declareFullCapacity(void)
{
	if(areYouSure())
	{
		capacityAs=fullCapacity*3600.0;
	}
}

void initSequence(void)
{
	initPowerSystem();
	initDelayTimer();
	initValues();
	initSampling();
	initDefaults();
	if((ReadEEPROM(0))!=0x76)
	{
		initRLE();
		saveSettings();
		WriteEEPROM(0, 0x76);
	} 
	else
	{
		restoreSettings();
	}
	updateSettings();
	//
	initBeep();
	initPWM();
	initLCD();
	initADC();
	initKeys();
	initMenuStack();
	initKeypad();
	IPR2bits.USBIP=1;	   // select interrupt priority
	UIR=0;
	PIR2bits.USBIF=0;
	PIE2bits.USBIE=1;		// enable USB interrupts!	(added MG)
	RCONbits.IPEN=1;		// enable prioritized interrupts
	INTCONbits.GIEL=1;		// enable low interrupts
	INTCONbits.GIE=1;		// enable all interrupts
	USBConnect();
}

#pragma interruptlow myISR
void myISR(void)
{
	byte x;

	if(PIR2bits.TMR3IF & PIE2bits.TMR3IE)
	{
		if(!alarmOn)BEEP=0;
		scanCode++;
		if(scanCode>=4)scanCode=0;
		if(brightness<setBrightness)putPWM(brightness+2); else if(brightness>setBrightness)putPWM(brightness-2);
		switch(scanCode)
		{
			case 0:
			default:
				scanLn=~0x10;
				break;
			case 1:
				scanLn=~0x20;
				break;
			case 2:
				scanLn=~0x40;
				break;
			case 3:
				scanLn=~0x80;
				break;
	}
	PORTB=(scanLn);
	_asm
		nop
		nop
		nop
		nop
	_endasm
	x=PORTB;	
	x=(~x & 0x0F);
	if((x!=0)&&(noKeys))
	{
		scanCounter=KEY_HOLD_OFF;
		switch(x)
		{
			case 0x01:
				noKeys=0;
				scanRaw=scanCode;
				putKey(scanRaw);
				BEEP=1;
				break;
			case 0x02:
				noKeys=0;
				scanRaw=scanCode+4;
				putKey(scanRaw);
				BEEP=1;
				break;
			case 0x04:
				noKeys=0;
				scanRaw=scanCode+8;
				putKey(scanRaw);				
				BEEP=1;
				break;
			case 0x08:
				noKeys=0;
				scanRaw=scanCode+12;
				putKey(scanRaw);
				BEEP=1;
				break;
		}
	} else
	if(x==0)
	{
	if(scanCounter>0)scanCounter--; else noKeys=1;
	}
	PIR2bits.TMR3IF=0;
	}

	if(INTCONbits.TMR0IF & INTCONbits.TMR0IE)
	{
		if(timerDelay!=0)timerDelay--;
		INTCONbits.TMR0IF=0;
	}
}

void updateValueString(byte index, ram char *outstr)
{
	switch(index)
	{	
		case LOG_BATTV:
				*outstr++='B';
				*outstr++='a';
				*outstr++='t';
				*outstr++='t';
				*outstr++='.';
				*outstr++='V';
				break;
		case LOG_CIRCUITV:
				*outstr++='C';
				*outstr++='i';
				*outstr++='r';
				*outstr++='c';
				*outstr++='.';
				*outstr++='V';
				break;
		case LOG_LOADA:
				*outstr++='L';
				*outstr++='o';
				*outstr++='a';
				*outstr++='d';
				*outstr++='.';
				*outstr++='A';
				break;
		case LOG_CHARGEA:
				*outstr++='C';
				*outstr++='h';
				*outstr++='r';
				*outstr++='g';
				*outstr++='.';
				*outstr++='A';
				break;
		case LOG_CIRCUITA:
				*outstr++='C';
				*outstr++='i';
				*outstr++='r';
				*outstr++='c';
				*outstr++='.';
				*outstr++='A';
				break;
		case LOG_NETA:
				*outstr++='N';
				*outstr++='e';
				*outstr++='t';
				*outstr++='.';
				*outstr++='A';
				break;
		case LOG_CAPACITYAH:
				*outstr++='C';
				*outstr++='p';
				*outstr++='t';
				*outstr++='y';
				*outstr++='.';
				*outstr++='A';
				*outstr++='H';
				break;
		case LOG_CAPACITYPERCENT:
				*outstr++='C';
				*outstr++='p';
				*outstr++='t';
				*outstr++='y';
				*outstr++='.';
				*outstr++='%';
				break;
		default:
				*outstr++='N';
				*outstr++='o';
				*outstr++='n';
				*outstr++='e';
				break;
	}
	*outstr='\0';
}

void updateLogChannelStrings(byte x)
{
	byte i;
	for(i=0; i<RLE_CHANNELS; i++)
	{
	updateValueString(logChannel[i], &logChannelString[i][0]);
	}
}

double getValue(byte index)
{
	double f;

	switch(index)
	{	
		case LOG_BATTV:
				f=batteryVoltage;
				break;
		case LOG_CIRCUITV:
				f=circuitVoltage;
				break;
		case LOG_LOADA:
				f=loadCurrent;
				break;
		case LOG_CHARGEA:
				f=chargeCurrent;
				break;
		case LOG_CIRCUITA:
				f=circuitCurrent;
				break;
		case LOG_NETA:
				f=netCurrent;
				break;
		case LOG_CAPACITYAH:
				f=batteryCapacityAH;
				break;
		case LOG_CAPACITYPERCENT:
				f=batteryPercent;
				break;
		default:
				f=0.0;
				break;
	}
	return f;
}

#pragma interrupt myISRHigh
void myISRHigh(void)
{
	int x;
	byte i;
	double f;
	static byte stby;
	static char c;
	char d;
	byte j;
	//

	if(PIR1bits.TMR1IF & PIE1bits.TMR1IE)
	{
	if(samplingIndex<(NUM_SAMPLES-1))samplingIndex++; else samplingIndex=0;
	if(samplingCount<(SAMPLING_COUNT-1))samplingCount++; else samplingCount=0;
	i=samplingIndex;
	// now update samples...
	switch(i)
	{
			case ANCH_VBATTNEG:
			case ANCH_VUSB:
			case ANCH_VBATTPOS:
			case ANCH_INEG:
			case ANCH_IPOS:
					sample[i].updated=0;													// clear updated flag, ie, data in process of being changed!
					//sample[i].lastSampled=samplingCount;
					//sample[i].delta=sample[i].value;
					sample[i].value=sample[i].voltageDivider*(((double)(readAN(i)))*voltageRail50/1023.0);			// convert ADC value to a voltage in Volts
					//sample[i].delta=(sample[i].value-sample[i].delta);
					sample[i].avgValue=(sample[i].avgValue+sample[i].value)/2.0;			// compute average value
					sample[i].updated=1;													// done!
					break;

			case COMPUTE_STATE:
					// this is the last state before the cycle repeats 
					// In this state, we compute all totals...
					totalUpdated=0;
					// Primary Values
					batteryVoltage=sample[ANCH_VBATTPOS].avgValue;
					circuitVoltage=sample[ANCH_VBATTNEG].avgValue;
					loadDiffAmpVoltage=sample[ANCH_INEG].avgValue;
					chargeDiffAmpVoltage=sample[ANCH_IPOS].avgValue;
					vusbRail=sample[ANCH_VUSB].avgValue;
					// Secondary Values
					f=(1000.0/(shuntResistance*differentialGain))*sample[ANCH_INEG].avgValue;
					loadCurrent=(loadCurrent+f)/2.0;
					f=(1000.0/(shuntResistance*differentialGain))*sample[ANCH_IPOS].avgValue;
					chargeCurrent=(chargeCurrent+f)/2.0;
					f=1000.0*((sample[ANCH_VBATTPOS].avgValue-sample[ANCH_VBATTNEG].avgValue)/senseResistance);
					circuitCurrent=(circuitCurrent+f)/2.0;
					f=(chargeCurrent-loadCurrent-(circuitCurrent/1000.0));
					netCurrent=(netCurrent+f)/2.0;
					if(chargeCurrent>=0.001)
					{
						deltaCapacityAs=chargeCurrent*(chargingEfficiency/100.0)*(DELTAT);
						capacityAs+=deltaCapacityAs;
						batteryMinutes+=(deltaCapacityAs/(chargeCurrent*60.0));
					}
					else
					if(netCurrent<=-0.001)
					{
						f=exp(peukertsConstant*(log(-netCurrent)));
						deltaCapacityAs=-f*(DELTAT);
						capacityAs+=deltaCapacityAs;
						batteryMinutes=capacityAs/(60.0*f);
					}
					batteryCapacityAH=capacityAs/3600.0;
					batteryPercent=100.0*(batteryCapacityAH/fullCapacity);
					if(batteryPercent<(alarmCapacity-0.5))
					{
						if(samplingCount & 0x400)
						{
						if(BEEP==0)alarmOn|=1;
						} else
						{
							alarmOn&=~1;
						}
					} else alarmOn&=~1;
					if((alarmOn!=0)&&(beepOn==1))BEEP=1; else BEEP=0;
					time+=(DELTAT);				// accumulate time in seconds
					checkStandBy();
					if((standBy==0)||(BUTTON==0))
					{
						if(BUTTON==0)
						{
							putPWM(2.55*brightnessPercent);
							if(stby!=0)
							{
								stby=0;
								updateLCD();
							}
						} else
						{
							if(timeOutSecondsCounter>0.0)
							{
								timeOutSecondsCounter-=(DELTAT);
								if(timeOutSecondsCounter<=0.0)
								{
								setBrightness=0;
								} 
							}
						}
						(void)usbSense();
						stby=0;
					} else
					{
						putPWM(0);
						if(((samplingCount & 0x200)==0)&&(stby!=2))
						{
							stby=2;						
							if(c=='m')
							{
								c='%'; d=' ';
								f=batteryPercent;
							}
							else
							{
								c='m'; d='A';
								f=circuitCurrent;
							}
							writeStringLCDDirect(0x80, (rom char*)"  Standby Mode  ");
							writeStringLCDDirect(0xC0, (rom char*)"                ");
							writeLCD(0xC4, LCDCOMMAND);
							if(f<=0.0)
							{
							f=-f;
							writeLCD('-', LCDDATA);
							}
							x=printFDecimal(f, 1, 0);
							while(buffer[x]!='\0')writeLCD(buffer[x++], LCDDATA);
							writeLCD(' ', LCDDATA);
							writeLCD(c, LCDDATA);
							writeLCD(d, LCDDATA);
						} else
						if(((samplingCount & 0x200)!=0)&&(stby!=1))
						{
							stby=1;
							writeStringLCDDirect(0x80, (rom char*)"                ");
							writeStringLCDDirect(0xC0, (rom char*)"                ");
						}
					}
					totalUpdated=1;
					break;

			case RLE_STATE:
					if(logPeriodCounter>0.0)logPeriodCounter-=DELTAT;
					else
					{	
						logPeriodCounter=logPeriod;
						for(i=0; i<RLE_CHANNELS; i++)
						{
							j=logChannel[i];
							if(j!=0)
							{
								pushRLEData(i, getValue(j), logError);
							}					
						}
					}
					break;
	}
	PIR1bits.TMR1IF=0;
	}

	if(UIR & UIE)
	{
	// this service routine must be in the same priority as TMR1. This is to avoid
	// incoherencies in the RLE buffer that TMR1 ISR writes and the USB ISR reads...
	USBDriverService();
	BootService();
	PIR2bits.USBIF=0;
	}

}

/*
int waitUpdated(byte index)
{
		// blocks until the sample i has been updated and returns with maximum time before next update
		while(samplingIndex!=index);
		while(sample[index].updated==0);
		return 1;
}
*/

double ReadEEPROMF(int address)
{
	// get a floating point from EEPROM memory at address address
	// 4 bytes long (float)
	double x;
	byte* ptr;

	x=0.0;
	ptr=(byte*)&x;	
	*ptr=ReadEEPROM(address);
	*(ptr+1)=ReadEEPROM(address+1);
	*(ptr+2)=ReadEEPROM(address+2);
	*(ptr+3)=ReadEEPROM(address+3);
	return x;
}

void WriteEEPROMF(int address, double f)
{
	// store a floating point double to EEPROM memory at address address
	// 4 bytes long (float)
	byte* ptr;

	ptr=(byte*)&f;	
	WriteEEPROM(address, (byte)(*ptr));
	WriteEEPROM(address+1, (byte)*(ptr+1));
	WriteEEPROM(address+2, (byte)*(ptr+2));
	WriteEEPROM(address+3, (byte)*(ptr+3));
}

#define NUM_SETTINGS	19
#define SETTINGS_OFFSET	16

double* settingsList[NUM_SETTINGS]=
{
	&differentialGain,
	&shuntResistance,
	&senseResistance,
	&timeOutInSeconds,
	&chargingEfficiency,
	&fullCapacity,
	&voltageRail33,
	&voltageRail50,
	&dividerLow,
	&dividerHigh,
	&peukertsConstant,
	&minVoltage,
	&maxVoltage,
	&brightnessPercent,
	&cutOffVoltage,
	&alarmCapacity,
	&logPeriod,
	&logError,
	&currentThreshold
};

#define NUM_BYTESETTINGS	8
#define BYTESETTINGS_OFFSET	240

byte* byteSettingsList[NUM_BYTESETTINGS]=
{
	&displayMode,						// must be first!
	&roundingMode,
	&beepOn,
	&cellType,
	&logChannel[0],
	&logChannel[1],
	&logChannel[2],
	&logChannel[3]
};

void saveSettings(void)
{
	byte i;
	byte offset;

	offset=SETTINGS_OFFSET;
	for(i=0; i<NUM_SETTINGS; i++)
	{
		WriteEEPROMF(offset, (*settingsList[i]));
		offset+=sizeof(double);
	}

	offset=BYTESETTINGS_OFFSET;
	for(i=0; i<NUM_BYTESETTINGS; i++)
	{
		WriteEEPROM(offset, (*byteSettingsList[i]));
		offset+=sizeof(byte);
	}
}

void restoreSettings(void)
{
	byte i;
	byte offset;
	
	offset=SETTINGS_OFFSET;
	for(i=0; i<NUM_SETTINGS; i++)
	{
		*(settingsList[i])=ReadEEPROMF(offset);
		offset+=sizeof(double);
	}

	offset=BYTESETTINGS_OFFSET;
	for(i=0; i<NUM_BYTESETTINGS; i++)
	{
		*(byteSettingsList[i])=ReadEEPROM(offset);
		offset+=sizeof(byte);
	}
}

byte getNextMode(byte thismode)
{
	byte nmode;
	if((thismode>='1')&&(thismode<='8'))nmode=thismode+1;
	else
	if((thismode>='A')&&(thismode<='C'))nmode=thismode+1;
	else
	if(thismode=='9')nmode='A';
	else
	if(thismode=='*')nmode='1';
	else
	if(thismode=='D')nmode='*';
	else nmode='1';
	return nmode;
}

void displayDisplayMode(byte mode)
{
	static byte lastmode;

	switch(mode)
	{
		case MODE_CURRENT:
			seeChargeCurrentAndLoadCurrent(0);
			lastmode=mode;
			break;
		case MODE_TIME_REMAINING:
			seeBatteryChargeAndMinutes(0);
			lastmode=mode;
			break;
		case MODE_VOLTAGE_CURRENT:
			seeBatteryVoltageCurrent(0);
			lastmode=mode;
			break;
		case MODE_CAPACITY:
			seeCapacityDouble(0);
			lastmode=mode;
			break;
		case MODE_ALL_SCROLLING:
			seeVoltages(0);
			lastmode=mode;
			break;
		case MODE_CIRCUIT:
			seeCircuit(0);
			lastmode=mode;
			break;
		case SINGLE_MODE_CHARGE:
			seeBatteryPercent(0);
			lastmode=mode;
			break;
		case SINGLE_MODE_CAPACITYAH:
			seeBatteryCapacityAH(0);
			lastmode=mode;
			break;
		case SINGLE_MODE_VOLTAGE:
			seeBatteryVoltage(0);
			lastmode=mode;
			break;
		case SINGLE_MODE_LOADCURRENT:
			seeLoadCurrent(0);
			lastmode=mode;
			break;
		case SINGLE_MODE_CHARGECURRENT:
			seeChargeCurrent(0);
			lastmode=mode;
			break;
		case SINGLE_MODE_CIRCUITCURRENT:
			seeCircuitCurrent(0);
			lastmode=mode;
			break;
		case SINGLE_MODE_NETBATTERYDRAIN:
			seeNetBatteryDrain(0);
			lastmode=mode;
			break;
		case SINGLE_MODE_TIMEREMAINING:
			seeBatteryTimeMinutes(0);
			lastmode=mode;
			break;
		case SINGLE_MODE_ALL:
			lastmode=getNextMode(lastmode);
			displayDisplayMode(lastmode);
			break;
	}
}

void main(void)
{
	 byte key;
	 byte scrollingMode;

	 TRISC=INIT_TRISC;
	 TRISB=INIT_TRISB;
	 PORTC=0;
	 PORTB=0;
  	 
	 delayMs(250);
	 initSequence();
	 writeStringLCD(0, (rom char*)"Battery CapacityMeter ver.");
	 disFUnsignedLCD(26, firmwareVersion, 2, 0, 4);
	 delayMs(1000);
	 clearLCD();
	 while(1)
	 {
			if(keyFull==0)
			{
			displayDisplayMode(displayMode);
			timerDelay=1;
			while(timerDelay>0)goToLowPowerMode();
			delayMs(500);
			if(displayMode==SINGLE_MODE_ALL)
				{
				timerDelay=2;
				while(timerDelay>0)goToLowPowerMode();
				//delayMs(2000);
				}
			} 
			else
			{
				key=getKey();
				if(key=='#')
				{
				enableStandByDetect=0;
				goOutOfStandBy();
				lcdTransitionType=1;
				updateLCD();
				initKeys();
				pushMenu((rom MENU*)&mainMenu, 0);
				executeMenuStack();
				lcdTransitionType=-1;
				updateLCD();
				enableStandByDetect=1;
				} else
				{
					displayMode=key;
					scrollingMode=key;
					WriteEEPROM(BYTESETTINGS_OFFSET, displayMode);
				}			
			}
			enableStandByDetect=1;
			if((batteryVoltage<cutOffVoltage)&&(USBSENSE==0))
			{
				enableStandByDetect=0;
				writeStringLCD(0, (rom char*) "Voltage Too Low ");
				writeStringLCD(16, (rom char*)"Shutting Down...");
				writeScreen(&lcdScreen[0]);
				delayMs(3000);
				clearLCD();
				powerOff();
			}
	 }
}

/** EOF main.c ***************************************************************/
